using System;
using System.Linq;
using System.ComponentModel.DataAnnotations;

namespace HIPS.Web.UI.ViewModels.AssistedRegistration
{
    public class UnregisteredPatientViewModel
    {
        #region Properties

        public string FamilyName { get; set; }

        public string GivenNames { get; set; }
        
        public string Ihi { get; set; }

        [DisplayFormat(DataFormatString = "dd/MM/yyyy")]
        public DateTime? DateOfBirth { get; set; }

        [Required]
        public string HospitalCode { get; set; }

        public string Ward { get; set; }

        public string Room { get; set; }

        public string Bed { get; set; }

        /// <summary>
        /// Gets the location (ward:room:bed) of the patient.
        /// </summary>
        public string Location
        {
            get
            {
                string location = String.Join(":",new [] { this.Ward, this.Room, this.Bed });
                return (location == "::" ? "Unknown" : location);
            }
        }

        /// <summary>
        /// Gets the IHI of the patient formatted for display.
        /// </summary>
        public string FormattedIhi
        {
            get
            {
                return System.Text.RegularExpressions.Regex.Replace(this.Ihi, @"(\d{4})(\d{4})(\d{4})(\d{4})", @"$1 $2 $3 $4");
            }
        }

        #endregion

        #region View Logic

        public RegistrationMethods ValidRegistrationMethods
        {
            get
            {
                // Unknown DOB (Unregistrable)
                if (!DateOfBirth.HasValue)
                {
                    return RegistrationMethods.None;
                }

                // Under 14
                if (DateOfBirth > DateTime.Now.AddYears(-14))
                {
                    return RegistrationMethods.DependantOnly;
                }

                // 14 or over, under 18
                if (DateOfBirth > DateTime.Now.AddYears(-18))
                {
                    return RegistrationMethods.StandardOrDependant;
                }

                // 18 or over
                return RegistrationMethods.StandardOnly;
            }
        }

        public enum RegistrationMethods
        {
            None,
            StandardOnly,
            DependantOnly,
            StandardOrDependant
        }

        public string DateOfBirthFormatted
        {
            get { return DateOfBirth.HasValue ? DateOfBirth.Value.ToString("dd/MM/yyyy") : String.Empty; }
        }

        #endregion

    }
}